--  usage: add a line like 
-- 
-- dofile("d:\\prj\\util\\wireshark-dissectors\\udp2av.lua")
--  or 
-- dofile("c:\\vision\\bin\\wireshark-dissectors\\udp2av.lua")
--
--  to the end of Wireshark's init.lua file, which is typically 
--
-- c:\Program Files\Wireshark\init.lua 
--
-- to enable LUA in Wireshark comment out the first line after the initial comments of init.lua
--
-- To enter the ports you use, from the Wireshark menu:
-- open Edit/Preferences and select Protocols/UDP2AV

do
	local p_udp2av = Proto("udp2av","alpha-bit Udp2Av")
	
	local ePacketTypes = {
		[0] = "NORMAL",
		[1] = "ROLLBACK",
		[2] = "LIFESIGN",
		[3] = "PACKETIDTOOLOW",
	}
	DATE_TIME = 3
	REPORT_HIGH = 4
	VALUES_HIGH = 5
	CONTROL_STATE = 7
	FADE = 8
	RECEIPT_ALARM = 9
	DYN_LIMIT = 12
	IMA_PARAMS = 16
	GENERAL_QUERY = 17 
	PO_STRING = 18
	CONTROL_VALUE = 23
	
	RELIABLE_HDR_LEN = 24
	RECVHEAD_LEN = 8
	SENDHEAD_LEN = 4
	
	local eTgrName = {
		[DATE_TIME] = "DATE_TIME",
		[REPORT_HIGH] = "REPORT_HIGH",
		[VALUES_HIGH] = "VALUES_HIGH",
		[CONTROL_STATE] = "CONTROL_STATE",
		[FADE] = "FADE",
		[RECEIPT_ALARM] = "RECEIPT_ALARM",
		[DYN_LIMIT] = "DYN_LIMIT",
		[IMA_PARAMS] = "IMA_PARAMS",
		[GENERAL_QUERY] = "GENERAL_QUERY",
		[PO_STRING] = "PO_STRING",
		[CONTROL_VALUE] = "CONTROL_VALUE",
	}
	
	local eTgrDescr = {
		[DATE_TIME] = "date and time change (PLC <-> OS)",
		[REPORT_HIGH] = "status and value change (PLC -> OS)",
		[VALUES_HIGH] = "value change (PLC -> OS)",
		[CONTROL_STATE] = "status setpoint (OS -> PLC)",
		[FADE] = "fade in/out (OS -> PLC)",
		[RECEIPT_ALARM] = "alarm acknowledge (OS -> PLC)",
		[DYN_LIMIT] = "dynamic limit change (PLC -> OS)",
		[IMA_PARAMS] = "limit and delay setpoint (PLC <-> OS)",
		[GENERAL_QUERY] = "query all values and states (OS -> PLC)",
		[PO_STRING] = "string change (PLC <-> OS)",
		[CONTROL_VALUE] = "value setpoint (OS -> PLC)",
	}
	
	local eRecLen = {
		[DATE_TIME] = 12,
		[REPORT_HIGH] = 8,
		[VALUES_HIGH] = 6,
		[CONTROL_STATE] = 4,
		[FADE] = 4,
		[RECEIPT_ALARM] = 2,
		[DYN_LIMIT] = 8,
		[IMA_PARAMS] = 12,
		[GENERAL_QUERY] = 2,
		[PO_STRING] = 4,
		[CONTROL_VALUE] = 6,
	}
	
	local eLimitSign = {
		[0] = "OLD",
		[1] = "DELAY (delay entries are valid)",
		[2] = "VAL (limit ('Val') entry is valid)",
		[3] = "BOTH (all entries are valid)"
	}
	
	local f_SenderId = ProtoField.int32("udp2av.SenderId","SenderId",base.DEC)
	local f_PacketId = ProtoField.int32("udp2av.PacketId","PacketId",base.DEC)
	local f_ReceiverId = ProtoField.int32("udp2av.ReceiverId","ReceiverId",base.DEC)
	local f_PacketType = ProtoField.int32("udp2av.PacketType","PacketType",base.DEC,ePacketTypes)
	local f_TotalPacketLength = ProtoField.int32("udp2av.TotalPacketLength","TotalPacketLength",base.DEC)
	local f_PartialPacketId = ProtoField.int32("udp2av.PartialPacketId","PartialPacketId",base.DEC)
	local f_RollbackReqId = ProtoField.int32("udp2av.RollbackReqId","RollbackReqId",base.DEC)
	
	local f_Dummy = ProtoField.string("udp2av.Dummy","Dummy")
	-- PLC -> OS
	local f_PlcRecvHead_AnzEntry = ProtoField.uint16("udp2av.AnzEntry","AnzEntry",base.DEC)
	local f_PlcRecvHead_Kind = ProtoField.uint16("udp2av.Kind","Kind (PLC->OS telegram type)",base.DEC,eTgrName)
	local f_PlcRecvHead_Timestamp = ProtoField.uint32("udp2av.Timestamp","Timestamp (time since last hour in 1/10 sec.)",base.DEC)
	-- OS -> PLC
	local f_PlcSendHead_Destination = ProtoField.uint8("udp2av.Destination","Destination",base.DEC)
	local f_PlcSendHead_TgrType = ProtoField.uint8("udp2av.TgrType","TgrType (OS->PLC telegram type)",base.DEC,eTgrName)
	local f_PlcSendHead_BruttoLen = ProtoField.uint16("udp2av.BruttoLen","BruttoLen (including the 4 bytes header)",base.DEC)
	-- DATE_TIME
	local f_DateTime_DateTime = ProtoField.bytes("udp2av.DateTime","DateTime")
	local f_DateTime_LTOffset = ProtoField.int32("udp2av.LTOffset","LTOffset",base.DEC)
	-- General
	local f_ObjNo = ProtoField.uint16("udp2av.ObjNo","ObjNo",base.DEC)
	local f_State = ProtoField.uint16("udp2av.State","State",base.DEC)
	local f_Val = ProtoField.uint32("udp2av.Val","Val",base.DEC)
	local f_Fill8 = ProtoField.uint8("udp2av.Fill8","Fill8",base.DEC)
	local f_Fill16 = ProtoField.uint16("udp2av.Fill16","Fill16",base.DEC)
	-- FADE
	local f_Fade = ProtoField.bool("udp2av.Fade","Fade")
	-- IMA_PARAMS, DYN_LIMIT
	local f_RelNo = ProtoField.uint8("udp2av.RelNo","RelNo",base.DEC)
	local f_Sign = ProtoField.uint8("udp2av.Sign","Sign",base.DEC,eLimitSign)
	local f_DelayIn = ProtoField.uint16("udp2av.DelayIn","DelayIn",base.DEC)
	local f_DelayOut = ProtoField.uint16("udp2av.DelayOut","DelayOut",base.DEC)
	-- PO_STRING
	local f_String = ProtoField.string("udp2av.String","String")
	-- Error handling
	local f_UnknownTgrType = ProtoField.uint8("udp2av.UnknownTgrType","UnknownTgrType (OS->PLC telegram type)")
	local f_UnknownKind = ProtoField.uint16("udp2av.UnknownKind","UnknownKind (PLC->OS telegram type)")
	
	p_udp2av.prefs["ports"] = Pref.string("UDP-Ports", "", "Example: '3000-3009, 3050, 3051, 3080-3089'")
	p_udp2av.prefs["objno"] = Pref.uint("Object Filter", 0, "Example: '3124'")
	
	local filter_objno = 0
	 
	p_udp2av.fields = { 
		f_SenderId, 
		f_PacketId, 
		f_ReceiverId, 
		f_PacketType, 
		f_TotalPacketLength, 
		f_PartialPacketId, 
		f_RollbackReqId, 
		
		f_PlcRecvHead_AnzEntry,
		f_PlcRecvHead_Kind,
		f_PlcRecvHead_Timestamp, 
		
		f_PlcSendHead_Destination, 
		f_PlcSendHead_TgrType, 
		f_PlcSendHead_BruttoLen, 
		
		f_DateTime_DateTime,
		f_DateTime_LTOffset,

		f_ObjNo,
		f_State,
		f_Val,
		
		f_Fade,
		
		f_RelNo,
		f_Sign,
		f_DelayIn,
		f_DelayOut,
		
		f_fill8,
		f_fill16,
		
		f_UnknownTgrType,
		f_UnknownKind,
		}
	function includeObj(buf, pos, kind)
		if filter_objno == 0 then 
			return true 
		end
		if kind == DATE_TIME or kind == GENERAL_QUERY then
			return false
		end
		local offset = 0
		if kind == PO_STRING then 
			offset = 2
		end
		return buf(pos+offset,2):uint() == filter_objno
	end
	function calcRecLen(buf, pos, kind)
		if kind == PO_STRING then 
			return eRecLen[kind] + buf(pos+2,2):uint()
		end
		return eRecLen[kind]
	end
	function dissectVal(range,t)
		local t_val = t:add(f_Val, range)
		t_val:append_text(" (float="..tostring(range:float())..")")
	end
	function dissectDateTime(buf,pos,t)
		local time = tostring(buf(pos+0,1):uint()).."-"..tostring(buf(pos+1,1):uint()).."-"..tostring(buf(pos+2,1):uint())
		time = time .. " " .. tostring(buf(pos+3,1):uint())..":"..tostring(buf(pos+4,1):uint())..":"..tostring(buf(pos+5,1):uint())
		local t_time = t:add(f_DateTime_DateTime,buf(pos+0,6))
		t_time:set_text("DateTime: "..time)
		t:add(f_DateTime_LTOffset,buf(pos+6,4))
		t:add(f_Fill16,buf(pos+10,2))
		--return eRecLen[DATE_TIME]
	end
	function dissectReportHigh(buf,pos,t)
		t:add(f_ObjNo, buf(pos+0,2))
		t:add(f_State, buf(pos+2,2))
		dissectVal(buf(pos+4,4), t)
		--return eRecLen[REPORT_HIGH]
	end
	function dissectValuesHigh(buf,pos,t)
		t:add(f_ObjNo, buf(pos+0,2))
		dissectVal(buf(pos+2,4), t)
		--return eRecLen[VALUES_HIGH]
	end
	function dissectControlState(buf,pos,t)
		t:add(f_ObjNo, buf(pos+0,2))
		t:add(f_State, buf(pos+2,2))
		--return eRecLen[CONTROL_STATE]
	end
	function dissectFade(buf,pos,t)
		t:add(f_ObjNo, buf(pos+0,2))
		t:add(f_Fade, buf(pos+2,1))
		t:add(f_Fill8, buf(pos+3,1))
		--return eRecLen[FADE]
	end
	function dissectReceiptAlarm(buf,pos,t)
		t:add(f_ObjNo, buf(pos+0,2))
		--return eRecLen[RECEIPT_ALARM]
	end
	function dissectDynLimit(buf,pos,t)
		t:add(f_ObjNo, buf(pos+0,2))
		t:add(f_RelNo, buf(pos+2,1))
		t:add(f_Fill8, buf(pos+3,1))
		dissectVal(buf(pos+4,4),t)
		--return eRecLen[DYN_LIMIT]
	end
	function dissectImaParams(buf,pos,t)
		t:add(f_ObjNo, buf(pos+0,2))
		t:add(f_RelNo, buf(pos+2,1))
		t:add(f_Sign, buf(pos+3,1))
		t:add(f_DelayIn, buf(pos+4,2))
		t:add(f_DelayOut, buf(pos+6,2))
		dissectVal(buf(pos+8,4),t)
		--return eRecLen[IMA_PARAMS]
	end
	function dissectGeneralQuery(buf,pos,t)
		t:add(f_Fill16, buf(pos+0,2))
		--return eRecLen[GENERAL_QUERY]
	end
	function dissectPoString(buf,pos,t)
		local len = buf(pos+0,2):uint()
		t:add(f_Len, buf(pos+0,2))
		t:add(f_ObjNo, buf(pos+2,2))
		t:add(f_String, buf(pos+4,len))
		--return eRecLen[PO_STRING]+len
	end
	
	function p_udp2av.dissector(buf,pkt,root) 
		local t_udp2av = root:add(p_udp2av,buf(0,-1))
		t_udp2av:add(f_SenderId,buf(0,4))
		t_udp2av:add(f_PacketId,buf(4,4))
		t_udp2av:add(f_ReceiverId,buf(8,4))
		t_udp2av:add(f_PacketType,buf(12,4))
		t_udp2av:add(f_TotalPacketLength,buf(16,4))
		t_udp2av:add(f_PartialPacketId,buf(20,4))
		if buf(12,4):uint() == 1 then --rollback request
			if buf(16,4):uint() ~= 4 then 
				t_udp2av:set_expert_flags(PI_MALFORMED, PI_WARN)
				t_udp2av:append_text("rollback requests must have 4 bytes of rollback req. id")
			else
				t_udp2av:add(f_RollbackReqId,buf(24,4))
			end
		elseif buf(16,4):uint() > 0 and buf:len() == buf(16,4):uint() + 24 then

			local t_dat = t_udp2av:add(buf(24,-1), "data")
			local kind = 0
			local count = 0
			local pos = 0
			if buf(0,4):uint() <= 100 then
				-- PLC -> OS
				-- header
				local posAnzEntry = 24
				local posKind = posAnzEntry + 2
				local posTimestamp = posKind + 2 
				t_dat:add(f_PlcRecvHead_AnzEntry,buf(posAnzEntry,2))
				kind = buf(posKind,2):uint()
				if		kind == DATE_TIME 
					or	kind == REPORT_HIGH 
					or	kind == VALUES_HIGH 
					or	kind == DYN_LIMIT 
					or	kind == IMA_PARAMS 
					or	kind == PO_STRING
						then 
					t_dat:add(f_PlcRecvHead_Kind,buf(posKind,2))
				else
					t_dat:add(f_UnknownKind,buf(posKind,2))
				end
				t_dat:add(f_PlcRecvHead_Timestamp,buf(posTimestamp,4))
				count = buf(posAnzEntry,2):uint()
				pos = RELIABLE_HDR_LEN + RECVHEAD_LEN
			else
				-- OS -> PLC
				-- header
				local posDestination = 24
				local posTgrType = posDestination + 1
				local posBruttoLen = posTgrType + 1 
				t_dat:add(f_PlcSendHead_Destination,buf(posDestination,1))
				kind = buf(posTgrType,1):uint()
				if		kind == DATE_TIME 
					or	kind == FADE 
					or	kind == RECEIPT_ALARM 
					or	kind == CONTROL_VALUE 
					or	kind == CONTROL_STATE 
					or	kind == IMA_PARAMS
					or	kind == GENERAL_QUERY
					or	kind == PO_STRING
						then 
					t_dat:add(f_PlcSendHead_TgrType,buf(posTgrType,1))
				else
					t_dat:add(f_UnknownTgrType,buf(posTgrType,1))
				end
				t_dat:add(f_PlcSendHead_BruttoLen,buf(posBruttoLen,2))
				count = (buf(posBruttoLen,2):uint()-4) / eRecLen[kind]
				pos = RELIABLE_HDR_LEN + SENDHEAD_LEN
				if (buf(posBruttoLen,2):uint()-4) % eRecLen[kind] ~= 0 then
					t_dat:append_text("BruttoLen does not match record length according to TgrType")
					t_dat:set_expert_flags(PI_MALFORMED, PI_WARN)
				end
			end
			local dissect = nil
			if kind == DATE_TIME then
				dissect = dissectDateTime
			elseif kind == REPORT_HIGH then
				dissect = dissectReportHigh
			elseif kind == VALUES_HIGH then
				dissect = dissectValuesHigh
			elseif kind == CONTROL_STATE then
				dissect = dissectControlState
			elseif kind == FADE then
				dissect = dissectFade
			elseif kind == RECEIPT_ALARM then
				dissect = dissectReceiptAlarm
			elseif kind == DYN_LIMIT then
				dissect = dissectDynLimit
			elseif kind == IMA_PARAMS then
				dissect = dissectImaParams
			elseif kind == GENERAL_QUERY then
				dissect = dissectGeneralQuery
			elseif kind == PO_STRING then
				dissect = dissectPoString
			elseif kind == CONTROL_VALUE then
				dissect = dissectValuesHigh --same records structure
			else
				t_dat:append_text(" unknown telegram type "..tostring(kind))
				t_dat:set_expert_flags(PI_MALFORMED, PI_WARN)
			end
			if dissect then
				local reclen = calcRecLen(buf, pos, kind)
				if reclen then
					t_dat:set_text(eTgrName[kind].."("..eTgrDescr[kind]..")")
					for i = 1, count, 1 do
						if includeObj(buf, pos, kind) then
							t_msg = t_dat:add(buf(pos,reclen), eTgrName[kind].."["..tostring(i).."]")
							--t_msg:set_text()
							--t_msg:set_generated()
							dissect(buf, pos, t_msg)
						end
						pos = pos + reclen
					end
				end
			end

		end
	end
	
	
	function p_udp2av.init()
		filter_objno = p_udp2av.prefs.objno 
		if p_udp2av.prefs.ports ~= "" then
			local udp_encap_table = DissectorTable.get("udp.port")
			local ports = p_udp2av.prefs.ports
			for range in string.gmatch(ports, "%s*([^,]*)%s*") do
				portFrom,portTo = string.match(range, "%s*(%d*)%s*\-%s*(%d*)%s*")
				if portFrom and portTo then
					for port = tonumber(portFrom), tonumber(portTo), 1 do
						udp_encap_table:add(tonumber(port),p_udp2av)
					end
				elseif range and tonumber(range) then
					udp_encap_table:add(tonumber(range),p_udp2av)
				end		       
			end
		end
	end
end
